// ============================================================================
// ============================================================================
// ============================================================================
// ==                                                                        ==
// == Name    : TheEmuLib.Emu_Adjust_RGBA_BC.fsh                             ==
// == Type    : Fragment shader                                              ==
// == Version : 1.0.1 (2017/02/13)                                           ==
// == Creator : TheEmu © TheEmu 2017, Some Rights Reserved                   ==
// == Licence : Creative Commons Attribution-ShareAlike 4.0                  ==
// ==           http://creativecommons.org/licences/by-sa/4.0                ==
// ==                                                                        ==
// == Purpose: To adjust the brightness and contrast of an image.            ==
// ==                                                                        ==
// == Description: The brightness and/or contrast of the red, green and blue ==
// == components of an image's colour are adjusted by factors  specified  by ==
// == control parameters input to the shader as uniform variables.  Each  of ==
// == the R, G and B channels may be adjusted separately.  The alpha channel ==
// == is also subject to the same adjustment process but it is expected that ==
// == normaly the adjustment factor will be specified as 0 for this channel. ==
// ==                                                                        ==
// == For the brightness adjustment linear interpolation between the current ==
// == value and either 0.0 or 1.0 is used depending on whether brightness is ==
// == to be increased or decreased.                                          ==
// ==                                                                        ==
// == For the contrast adjustments the colour values are interpolated with a ==
// == sigmoidal weighting factor.                                            ==
// ==                                                                        ==
// == This file is a member of The Emu's shader library.                     ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == Update history:                                                        ==
// ==                                                                        ==
// ==   2016/12/26 - v1.0.0 - Initial version                                ==
// ==   2017/02/13 - v1.0.1 - Corrected spelling for normalisation function. ==
// ==                                                                        ==
// ============================================================================
// ============================================================================
// ============================================================================

// ============================================================================
// == Standard shader inputs ==================================================
// ============================================================================

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// The image that is to be manipulated.

uniform sampler2D iChannel0;

// ============================================================================
// == Imports from TheEmuLib ==================================================
// ============================================================================
//
// The GLSL shader language currently provides no mechanism for importing  any
// elements that are defined in other modules, not even C's crude source level
// #include mechanism. In the absence of anything better TheEmuLib handles any
// imports by manualy copying relevent utility code snippets from the  sources
// in the Shader Lib.Inc directory. This is very crude but I have attempted to
// be systematic in the way in which this is presented in the library sources.
//
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Basic_Constants.lib.src

#define PI_by_4 0.78539816339744830961566084581988

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Common_Utilities.lib.src

#define EMU_DEFAULT(type,x,default_colour) (x==type(0.0)) ? (default_colour) : (x)

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Functions from TheEmuLib.Emu_Coordinate_Normalisation.lib.src

vec2 Emu_Normalise_to_Window ( vec2 xy ) { return xy / u_WindowSize.xy; }

// ============================================================================
// == Shader specific inputs ==================================================
// ============================================================================

// The adjustment factors for brightness and contrast are each specified using
// the following parameters which specify separate factors for the red, green,
// blue and alpha colour channels.  The individual factors may range from -1.0
// to +1.0 with 0.0 corresponding to no change. All adjustment factors default
// to 0.0.  The effects of adjustment factors outside the range -1.0 to 1.0 is
// undefined.
//
// It is expected that in normal use the red, green and blue channels will  be
// adjusted equaly and that the alpha channel adjustment will be 0.0. 

uniform vec4 Emu_Adjust_RGBA_BC_brighten;
uniform vec4 Emu_Adjust_RGBA_BC_contrast;

vec4 brighten = clamp ( Emu_Adjust_RGBA_BC_brighten, vec4(-1.0), vec4(1.0) );
vec4 contrast = clamp ( Emu_Adjust_RGBA_BC_contrast, vec4(-1.0), vec4(1.0) );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The following parameter is used to control how the adjustment is performed.
// Currently there is only one aspect to this control which is specified using
// the least significant digit of the parameter,  if this is 1 then luminosity
// is preserved by the adjustment.

uniform int Emu_Adjust_RGBA_BC_mode;

bool constant_luminosity = Emu_Adjust_RGBA_BC_mode == 1; 

// ============================================================================
// == The shader's main routine ===============================================
// ============================================================================

void main ( void )
 {
   // Get the normalised coordinates of the current point.

   vec2 uv = Emu_Normalise_to_Window ( gl_FragCoord.xy );

   // Look up the corresponding colour in the source image.

   vec4 colour = texture2D ( iChannel0, uv );

   // Remember how bright it was.

   float old_luminocity = length(colour.rgb);

   // Adjust colour component c by brightness factor b using
   // linear interpolation between c and either 0.0  or  1.0
   // depending on whether we are increasing  or  decreasing 
   // the brightness, i.e. using
   //
   //       c = c +      c  * b   when b < 0.0
   //       c = c + ( (1-c) * b ) when b > 0.0
   //
   // which is equivalent to the expression used here

   colour -= colour*abs(brighten) - max(brighten,0.0);

   // Adjust the contrast using a sigmoid interpolation. The
   // exact form of the sigmoid matters little, we use tan.

   vec4 a = ( 1.0 + contrast ) * PI_by_4;
   colour = ( colour - 0.5 ) * tan(a) + 0.5;  

   // Optionaly restore the old overall luminosity.

   if ( constant_luminosity )
    { colour.rgb *= old_luminocity / length(colour.rgb);
    }

   // Update shader outputs.

   gl_FragColor = colour * gl_Color;

}